/**HEADER********************************************************************
 *
 * Copyright (c) 2010, 2013-2015 Freescale Semiconductor;
 * All Rights Reserved
 *
 *
 ***************************************************************************
 *
 * THIS SOFTWARE IS PROVIDED BY FREESCALE "AS IS" AND ANY EXPRESSED OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL FREESCALE OR ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 *
 **************************************************************************
 *
 * $FileName: audio_generator.c$
 * $Version :
 * $Date    :
 *
 * Comments:
 *
 * @brief  The file emulates a generator.
 *****************************************************************************/

/******************************************************************************
 * Includes
 *****************************************************************************/

#include "board.h"
#include "fsl_os_abstraction.h"
#include "fsl_device_registers.h"
#include "fsl_interrupt_manager.h"
#include "fsl_clock_manager.h"
#include "accel.h"

#include "project.h"

////////////////////////////////////////////////////////////////////////////////
// Variables
////////////////////////////////////////////////////////////////////////////////

    accel_dev_t accDev;
    accel_dev_interface_t accDevice;
    accel_sensor_data_t accelData;
    accel_i2c_interface_t i2cInterface;

#include "cp.h"

void ftm_init_app_layer (void)
{
    // Register callback func for I2C
    i2cInterface.i2c_init       =  I2C_DRV_MasterInit;
    i2cInterface.i2c_read       =  I2C_DRV_MasterReceiveDataBlocking;
    i2cInterface.i2c_write      =  I2C_DRV_MasterSendDataBlocking;

    accDev.i2c = &i2cInterface;
    accDev.accel = &accDevice;

    accDev.slave.baudRate_kbps  = BOARD_ACCEL_BAUDRATE;
    accDev.slave.address        = BOARD_ACCEL_ADDR;
    accDev.bus                  = BOARD_ACCEL_I2C_INSTANCE;

    // Initialize standard SDK demo application pins.

    // Initialize the Accel.
    if (accel_init(&accDev)) STATIC_SANITY_CHECK;

}

#define ADRS_VALID 0x5432

#define DEFAULTXTHRESHOLD 5
#define DEFAULTYTHRESHOLD 5

enum
{
    XNOCHANGE = 0,
    XHASINCREASED,
    XHASDECREASED,
    YNOCHANGE,
    YHASINCREASED,
    YHASDECREASED
};

typedef struct 
{
    int16_t m_X,m_Y,m_XThreshold,m_YThreshold,m_ValidSig,m_LastActionX,m_LastActionY;
}    ACCELEROMETERDELTARECORDSTRUCT;

ACCELEROMETERDELTARECORDSTRUCT g_ADRS;

CP_STATUSCODE accelerometer_IncomingSingleTLVCallback (unsigned long p_Tag, unsigned long p_Len,unsigned char *p_MarshalledVal)
{
    unsigned long a_Value = (ntohl(*((unsigned long *)p_MarshalledVal)));
    int16_t a_Action = a_Value >> 16; 
    switch (a_Action)
    {
        case XNOCHANGE: LED1_OFF; break;
        case XHASINCREASED: LED1_ON; break;
        case XHASDECREASED: LED1_ON; break;
        case YNOCHANGE: LED2_OFF; LED3_OFF; break;
        case YHASINCREASED: LED2_ON; break;
        case YHASDECREASED: LED3_ON; break;
    }
    return CP_STATUSCODE_SUCCESS;
}

void RecordAccelerometerDeltaSingle(CP_COMMHANDLE p_CommHandle,uint16_t p_ModifiedValue,uint16_t p_Action)
{
#ifdef LOCAL_PROCESSING
    switch (p_Action)
    {
        case XNOCHANGE: LED1_OFF; break;
        case XHASINCREASED: LED1_ON; break;
        case XHASDECREASED: LED1_ON; break;
        case YNOCHANGE: LED2_OFF; LED3_OFF; break;
        case YHASINCREASED: LED2_ON; break;
        case YHASDECREASED: LED3_ON; break;
    }
#else
    // TODO: Error handling!
    cp_QueuePacketToComm(p_CommHandle,TAG_RESCHANGENOTIFICATION,(p_Action << 16) + p_ModifiedValue,CP_PRIQUEUE_LOWPRI);
#endif    
}        

void RecordAccelerometerDelta(CP_COMMHANDLE p_CommHandle,int16_t xData,int16_t yData)
{
    // on first invocation, record initial values so we won't get false positives.
    if (g_ADRS.m_ValidSig != ADRS_VALID)
    {
        g_ADRS.m_X = xData;
        g_ADRS.m_Y = yData;
        g_ADRS.m_XThreshold = DEFAULTXTHRESHOLD;
        g_ADRS.m_YThreshold = DEFAULTYTHRESHOLD;
        g_ADRS.m_ValidSig = ADRS_VALID;
        g_ADRS.m_LastActionX = XNOCHANGE;
        g_ADRS.m_LastActionY = YNOCHANGE;
    }
    else
    {
#ifdef STRESSTEST
        int16_t xDelta = rand();
        int16_t yDelta = rand();
#else
        int16_t xDelta = (g_ADRS.m_X - xData);
        int16_t yDelta = (g_ADRS.m_Y - yData);
#endif        
        uint16_t a_XAction,a_YAction;
        // TODO: First process Deltas, then record new values
        if (xDelta < (g_ADRS.m_XThreshold * -1))
            a_XAction = XHASINCREASED;
        else
        {
            if (xDelta > g_ADRS.m_XThreshold)
                a_XAction = XHASDECREASED;
            else
                a_XAction = XNOCHANGE;
        }                
        // in order to reduce traffic, we only record the first of each NOCHANGE messages.
        if ((a_XAction != XNOCHANGE) || (g_ADRS.m_LastActionX != XNOCHANGE))
            RecordAccelerometerDeltaSingle(p_CommHandle,(uint16_t)xData,a_XAction);

        if (yDelta < (g_ADRS.m_YThreshold * -1))
            a_YAction = YHASINCREASED;
        else
        {
            if (yDelta > g_ADRS.m_YThreshold)
                a_YAction = YHASDECREASED;
            else
                a_YAction = YNOCHANGE;
        }

        if ((a_YAction != YNOCHANGE) || (g_ADRS.m_LastActionY != YNOCHANGE))
            RecordAccelerometerDeltaSingle(p_CommHandle,(uint16_t)yData,a_YAction);

        g_ADRS.m_X = xData;
        g_ADRS.m_Y = yData;
        g_ADRS.m_LastActionX = a_XAction;
        g_ADRS.m_LastActionY = a_YAction;
    }            
}

void sample_accelerometer_task (void *p_Param)
{
    CP_COMMHANDLE a_CommHandle = (CP_COMMHANDLE)p_Param;
    int16_t xData, yData;    

    ftm_init_app_layer(); // most do this here as the SPI comm code requires presence of OS.

    // Main loop.  Get sensor data and update globals for the FTM timer update.
    while(1)
    {
        // Wait 5 ms in between samples (accelerometer updates at 200Hz).
        vTaskDelay(5);

        // Get new accelerometer data.
        accDev.accel->accel_read_sensor_data(&accDev,&accelData);

        // Turn off interrupts (FTM) while updating new duty cycle values.
        // Get the X and Y data from the sensor data structure.
        xData = (int16_t)((accelData.data.accelXMSB << 8) | accelData.data.accelXLSB);
        yData = (int16_t)((accelData.data.accelYMSB << 8) | accelData.data.accelYLSB);

        // Convert raw data to angle (normalize to 0-90 degrees).  
        xData = (int16_t)(xData * 0.011);
        yData = (int16_t)(yData * 0.011);
        
        RecordAccelerometerDelta(a_CommHandle,xData,yData);
    }
}

/* EOF */
